# -*- coding: utf-8 -*-
"""
 __
|  \   _  _  _|_
|__/\/| )(_|_)|_\/
    /           /
"""

from resources.lib.modules import control, log_utils
from sys import version_info, platform as sys_platform
from threading import Thread
import time
from datetime import timedelta
import os
import shutil
import hashlib
window = control.homeWindow
pythonVersion = '{}.{}.{}'.format(version_info[0], version_info[1], version_info[2])
plugin = 'plugin://plugin.video.dynasty/'
LOGINFO = log_utils.LOGINFO
LOGDEBUG = log_utils.LOGDEBUG

properties = [
    'context.dynasty.settings',
    'context.dynasty.addtoLibrary',
    'context.dynasty.addtoFavourite',
    'context.dynasty.playTrailer',
    'context.dynasty.playTrailerSelect',
    'context.dynasty.traktManager',
    'context.dynasty.clearProviders',
    'context.dynasty.clearBookmark',
    'context.dynasty.rescrape',
    'context.dynasty.playFromHere',
    'context.dynasty.autoPlay',
    'context.dynasty.sourceSelect',
    'context.dynasty.findSimilar',
    'context.dynasty.browseSeries',
    'context.dynasty.browseEpisodes']

class CheckSettingsFile:
    def run(self):
        try:
            control.log('[ plugin.video.dynasty ]  CheckSettingsFile Service Starting...', LOGINFO)
            window.clearProperty('dynasty_settings')
            profile_dir = control.dataPath
            if not control.existsPath(profile_dir):
                success = control.makeDirs(profile_dir)
                if success: control.log('%s : created successfully' % profile_dir, LOGINFO)
            else: control.log('%s : already exists' % profile_dir, LOGINFO)
            settings_xml = control.joinPath(profile_dir, 'settings.xml')
            if not control.existsPath(settings_xml):
                control.setSetting('trakt.message2', '')
                control.log('%s : created successfully' % settings_xml, LOGINFO)
            else: control.log('%s : already exists' % settings_xml, LOGINFO)
            return control.log('[ plugin.video.dynasty ]  Finished CheckSettingsFile Service', LOGINFO)
        except:
            log_utils.error()

class FavouritesSync:
    def __init__(self):
        self.addon_data_path = control.dataPath
        self.favourites_db = control.joinPath(self.addon_data_path, 'favourites.db')
        # Determine download directory based on platform
        if sys_platform.startswith('win'):
            self.download_dir = os.path.join(os.path.expanduser('~'), 'Downloads')
        else:  # Android
            self.download_dir = '/storage/emulated/0/Download'
        self.download_favourites = os.path.join(self.download_dir, 'favourites.db')
        
    def sync_favourites(self):
        # Check if favourites.db backup is enabled
        if control.setting('favourites.backup.enable') != 'true':
            control.log('[ plugin.video.dynasty ] Favourites.db backup is disabled, skipping sync...', LOGINFO)
            return
        
        try:
            # Check if notifications for favourites.db are enabled
            notify = control.setting('favourites.backup.notifications') == 'true'
            
            # If addon_data favourites.db doesn't exist but download directory has it
            if not control.existsPath(self.favourites_db) and os.path.exists(self.download_favourites):
                control.log('[ plugin.video.dynasty ] Copying favourites.db from download directory to addon_data...', LOGINFO)
                shutil.copy2(self.download_favourites, self.favourites_db)
                # Show notification only if enabled
                if notify:
                    control.notification(message='Favourites database restored from download directory')
            
            # If addon_data favourites.db exists, copy to download directory
            elif control.existsPath(self.favourites_db):
                control.log('[ plugin.video.dynasty ] Copying favourites.db to download directory...', LOGINFO)
                os.makedirs(self.download_dir, exist_ok=True)
                shutil.copy2(self.favourites_db, self.download_favourites)
                # Show notification only if enabled
                if notify:
                    control.notification(message='Favourites database backed up to download directory')
                
        except:
            log_utils.error()

class FavouritesMonitor:
    def __init__(self):
        self.favourites_sync = FavouritesSync()
        self.favourites_path = control.joinPath(control.dataPath, 'favourites.db')
        self.last_hash = self.get_favourites_hash()
        
    def get_favourites_hash(self):
        try:
            if control.existsPath(self.favourites_path):
                with open(self.favourites_path, 'rb') as f:
                    return hashlib.md5(f.read()).hexdigest()
            return ''
        except:
            return ''
            
    def run(self):
        control.log('[ plugin.video.dynasty ] Favourites Monitor Service Starting...', LOGINFO)
        while not control.monitor.abortRequested():
            try:
                # Check if favourites.db backup is enabled before proceeding
                if control.setting('favourites.backup.enable') == 'true':
                    current_hash = self.get_favourites_hash()
                    if current_hash != self.last_hash and current_hash != '':
                        control.log('[ plugin.video.dynasty ] Favourites.db changed, syncing...', LOGINFO)
                        self.favourites_sync.sync_favourites()
                        self.last_hash = current_hash
                control.sleep(5000)  # Check every 5 seconds
            except:
                log_utils.error()
        control.log('[ plugin.video.dynasty ] Favourites Monitor Service Stopping...', LOGINFO)

class SettingsMonitor(control.monitor_class):
    def __init__(self):
        control.monitor_class.__init__(self)
        control.refresh_playAction()
        control.refresh_libPath()
        window.setProperty('dynasty.debug.reversed', str(control.setting('debug.reversed')))
        window.setProperty('dynasty.updateSettings', 'true')
        for id in properties:
            if control.setting(id) == 'true':
                window.setProperty(id, 'true')
        control.log('[ plugin.video.dynasty ]  Settings Monitor Service Starting...', LOGINFO)

    def onSettingsChanged(self):
        if window.getProperty('dynasty.updateSettings') != 'true':
            return control.log('[ plugin.video.dynasty ]  Settings Monitor is off.', LOGINFO)
        window.setProperty('dynasty.updateSettings','false')

        try:
            window.clearProperty('dynasty_settings')
        except:
            control.log('[ plugin.video.dynasty ]  Exception clearing settings property...', LOGDEBUG)

        try:
            control.refresh_playAction()
        except:
            control.log('[ plugin.video.dynasty ]  Exception making refreshing playAction...', LOGDEBUG)
        try:
            control.refresh_libPath()
        except:
            control.log('[ plugin.video.dynasty ]  Exception refreshing libpath...', LOGDEBUG)
        try:
            control.refresh_debugReversed()
        except:
            control.log('[ plugin.video.dynasty ]  Exception checking debug reversed', LOGDEBUG)
        try:
            control.setContextColors()
        except:
            control.log('[ plugin.video.dynasty ]  Exception setting context colors...', LOGDEBUG)
        try:
            control.checkModules()
        except:
            control.log('[ plugin.video.dynasty ]  Exception checking modules...', LOGDEBUG)
        try:
            control.sleep(50)
            refreshed = control.make_settings_dict()
            window.setProperty('dynasty.updateSettings','true')
        except:
            control.log('[ plugin.video.dynasty ]  Exception making settings dict...', LOGDEBUG)
        try:
            for id in properties:
                if control.setting(id) == 'true':
                    window.setProperty(id, 'true')
                else:
                    window.clearProperty(id)
        except:
            log_utils.error()

class SyncMyAccounts:
    def run(self):
        control.log('[ plugin.video.dynasty ]  Sync Accounts with Scraper Starting...', LOGINFO)
        control.syncAccounts()
        return control.log('[ plugin.video.dynasty ]  Finished Sync with Scraper', LOGINFO)

class SyncMovieLibrary:
    def run(self):
        if control.setting('library.cachesimilar') == 'true':
            control.log('[ plugin.video.dynasty ]  Sync Library Movies with Dynasty...', LOGINFO)
            from resources.lib.modules import library
            library.lib_tools().cacheLibraryforSimilar()
            return control.log('[ plugin.video.dynasty ]  Sync Library Movies with Dynasty Done', LOGINFO)
        else:
            return

class checkAutoStart:
    def run(self):
        control.log('[ plugin.video.dynasty ]  Checking for AutoStart....', LOGINFO)
        if control.setting('dynasty.autostart') == 'true':
            control.execute('RunAddon(plugin.video.dynasty)')
        return control.log('[ plugin.video.dynasty ]  Finished AutoStart Check', LOGINFO)

class ReuseLanguageInvokerCheck:
    def run(self):
        control.log('[ plugin.video.dynasty ]  ReuseLanguageInvokerCheck Service Starting...', LOGINFO)
        try:
            from xml.dom.minidom import parse as mdParse
            from resources.lib.modules.language_invoker import gen_file_hash
            addon_xml = control.joinPath(control.addonPath('plugin.video.dynasty'), 'addon.xml')
            current_addon_setting = control.addon('plugin.video.dynasty').getSetting('reuse.languageinvoker')
            try:
                tree = mdParse(addon_xml)
                reuse = tree.getElementsByTagName("reuselanguageinvoker")[0]
                current_xml_setting = reuse.firstChild.data
            except: 
                return control.log('[ plugin.video.dynasty ]  ReuseLanguageInvokerCheck failed to get settings.xml value', LOGINFO)
            if current_addon_setting == '':
                current_addon_setting = 'true'
                control.setSetting('reuse.languageinvoker', current_addon_setting)
            if current_xml_setting == current_addon_setting:
                return control.log('[ plugin.video.dynasty ]  ReuseLanguageInvokerCheck Service Finished', LOGINFO)
            control.okDialog(message='%s\n%s' % (control.lang(33023), control.lang(33020)))
            tree.getElementsByTagName("reuselanguageinvoker")[0].firstChild.data = current_addon_setting
            hash_start = gen_file_hash(addon_xml)
            newxml = str(tree.toxml())[22:]
            with open(addon_xml, "w") as f:
                f.write(newxml)
            hash_end = gen_file_hash(addon_xml)
            control.log('[ plugin.video.dynasty ]  ReuseLanguageInvokerCheck Service Finished', LOGINFO)
            if hash_start != hash_end:
                current_profile = control.infoLabel('system.profilename')
                control.execute('LoadProfile(%s)' % current_profile)
            else: 
                control.okDialog(title='default', message=33022)
            return
        except:
            log_utils.error()

class AddonCheckUpdate:
    def run(self):
        control.log('[ plugin.video.dynasty ]  Addon checking available updates', LOGINFO)
        try:
            import re
            import requests
            local_version = control.getDynastyVersion()
            if len(local_version) > 6:
                repo_xml = requests.get('https://www.midian.appboxes.co/wolfyB/Th3L48/nexus/zips/plugin.video.dynasty/addon.xml')
            else:
                repo_xml = requests.get('https://www.midian.appboxes.co/wolfyB/Th3L48/nexus/zips/plugin.video.dynasty/addon.xml')
            if not repo_xml.status_code == 200:
                return control.log('[ plugin.video.dynasty ]  Could not connect to remote repo XML: status code = %s' % repo_xml.status_code, LOGINFO)
            repo_version = re.findall(r'<addon id=\"plugin.video.dynasty\".+version=\"(\d*.\d*.\d*)\"', repo_xml.text)[0]
            def check_version_numbers(current, new):
                current = current.split('.')
                new = new.split('.')
                step = 0
                for i in current:
                    if int(new[step]) > int(i): return True
                    if int(i) > int(new[step]): return False
                    if int(i) == int(new[step]):
                        step += 1
                        continue
                return False
            if check_version_numbers(local_version, repo_version):
                while control.condVisibility('Library.IsScanningVideo'):
                    control.sleep(10000)
                control.log('[ plugin.video.dynasty ]  A newer version is available. Installed Version: v%s, Repo Version: v%s' % (local_version, repo_version), LOGINFO)
                control.notification(message=control.lang(35523) % repo_version)
            return control.log('[ plugin.video.dynasty ]  Addon update check complete', LOGINFO)
        except:
            log_utils.error()

class VersionIsUpdateCheck:
    def run(self):
        try:
            from resources.lib.database import cache
            isUpdate = False
            oldVersion, isUpdate = cache.update_cache_version()
            if isUpdate:
                window.setProperty('dynasty.updated', 'true')
                curVersion = control.getDynastyVersion()
                clearDB_version = '6.5.58'
                do_cacheClear = (int(oldVersion.replace('.', '')) < int(clearDB_version.replace('.', '')) <= int(curVersion.replace('.', '')))
                if do_cacheClear:
                    clr_fanarttv = False
                    cache.clrCache_version_update(clr_providers=False, clr_metacache=True, clr_cache=True, clr_search=False, clr_bookmarks=False)
                    from resources.lib.database import traktsync
                    clr_traktSync = {'bookmarks': False, 'hiddenProgress': False, 'liked_lists': False, 'movies_collection': False, 'movies_watchlist': False, 'popular_lists': False,
                                            'public_lists': False, 'shows_collection': False, 'shows_watchlist': False, 'trending_lists': False, 'user_lists': False, 'watched': False}
                    cleared = traktsync.delete_tables(clr_traktSync)
                    if cleared:
                        control.notification(message='Forced traktsync clear for version update complete.')
                        control.log('[ plugin.video.dynasty ]  Forced traktsync clear for version update complete.', LOGINFO)
                    if clr_fanarttv:
                        from resources.lib.database import fanarttv_cache
                        cleared = fanarttv_cache.cache_clear()
                        control.notification(message='Forced fanarttv.db clear for version update complete.')
                        control.log('[ plugin.video.dynasty ]  Forced fanarttv.db clear for version update complete.', LOGINFO)
                control.setSetting('trakt.message2', '')
                control.log('[ plugin.video.dynasty ]  Forced new User Data settings.xml saved', LOGINFO)
                control.log('[ plugin.video.dynasty ]  Plugin updated to v%s' % curVersion, LOGINFO)
        except:
            log_utils.error()

class SyncTraktCollection:
    def run(self):
        control.log('[ plugin.video.dynasty ]  Trakt Collection Sync Import Disabled...', LOGINFO)

class LibraryService:
    def run(self):
        try:
            library_hours = float(control.setting('library.import.hours'))
        except:
            library_hours = int(6)
        control.log('[ plugin.video.dynasty ]  Library Update Service Starting (Runs Every %s Hours)...' % library_hours,  LOGINFO)
        from resources.lib.modules import library
        library.lib_tools().service()

class SyncTraktService:
    def run(self):
        service_syncInterval = control.setting('trakt.service.syncInterval') or '15'
        control.log('[ plugin.video.dynasty ]  Trakt Sync Service Starting (sync check every %s minutes)...' % service_syncInterval, LOGINFO)
        from resources.lib.modules import trakt
        trakt.trakt_service_sync()

try:
    testDynasty = False
    kodiVersion = control.getKodiVersion(full=True)
    addonVersion = control.addon('plugin.video.dynasty').getAddonInfo('version')
    if len(str(control.getDynastyVersion())) > 6:
        repoVersion = control.addon('repository.dynastytest').getAddonInfo('version')
        repoName = 'repository.dynastytest'
        testDynasty = True
    else:
        try:
            repoVersion = control.addon('repository.dynasty').getAddonInfo('version')
            repoName = 'repository.dynasty'
        except:
            repoVersion = 'unknown'
            repoName = 'Unknown Repo'
    log_utils.log('########   CURRENT Dynasty VERSIONS REPORT   ########', level=LOGINFO)
    if testDynasty == True:
        log_utils.log('########   TEST Dynasty Version   ########', level=LOGINFO)
    log_utils.log('##   Platform: %s' % str(sys_platform), level=LOGINFO)
    log_utils.log('##   Kodi Version: %s' % str(kodiVersion), level=LOGINFO)
    log_utils.log('##   python Version: %s' % pythonVersion, level=LOGINFO)
    log_utils.log('##   plugin.video.dynasty Version: %s' % str(addonVersion), level=LOGINFO)
    log_utils.log('##   %s Version: %s' % (str(repoName), str(repoVersion)), level=LOGINFO)
    log_utils.log('######   DYNASTY SERVICE ENTERING KEEP ALIVE   #####', level=LOGINFO)
except:
    log_utils.log('## ERROR GETTING Dynasty VERSION - Missing Repo or failed Install ', level=LOGINFO)

def getTraktCredentialsInfo():
    username = control.setting('trakt.user.name').strip()
    token = control.setting('trakt.user.token')
    refresh = control.setting('trakt.refreshtoken')
    if (username == '' or token == '' or refresh == ''): return False
    return True

class PremAccntNotification:
    def run(self):
        from datetime import datetime
        from resources.lib.debrid import alldebrid
        from resources.lib.debrid import premiumize
        from resources.lib.debrid import realdebrid
        control.log('[ plugin.video.dynasty ] Debrid Account Expiry Notification Service Starting...', LOGINFO)
        self.duration = [(15, 10), (11, 7), (8, 4), (5, 2), (3, 0)]
        if control.setting('alldebridusername') != '' and control.setting('alldebridexpirynotice') == 'true':
            try:
                account_info = alldebrid.AllDebrid().account_info()['user']
            except:
                account_info = None
                from resources.lib.modules import log_utils
                log_utils.error()
            if account_info:
                if not account_info['isSubscribed']:
                    try:
                        expires = datetime.fromtimestamp(account_info['premiumUntil'])
                    except:
                        expires = datetime.today()-timedelta(days=1)
                        control.notification(message='AllDebrid Account has no expiration. Invalid or free account.', icon=control.joinPath(control.artPath(), 'alldebrid.png'))
                    days_remaining = (expires - datetime.today()).days
                    if days_remaining >= 0:
                        if self.withinRangeCheck('alldebrid', days_remaining):
                            control.notification(message='AllDebrid Account expires in %s days' % days_remaining, icon=control.joinPath(control.artPath(), 'alldebrid.png'))

        if control.setting('premiumizeusername') != '' and control.setting('premiumizeexpirynotice') == 'true':
            account_info = premiumize.Premiumize().account_info()
            if account_info:
                try:
                    expires = datetime.fromtimestamp(account_info['premium_until'])
                except:
                    expires = datetime.today()-timedelta(days=1)
                    control.notification(message='Premiumize.me Account has no expiration. Invalid or free account.', icon=control.joinPath(control.artPath(), 'premiumize.png'))
                days_remaining = (expires - datetime.today()).days
                if days_remaining >= 0:
                    if self.withinRangeCheck('premiumize', days_remaining):
                        control.notification(message='Premiumize.me Account expires in %s days' % days_remaining, icon=control.joinPath(control.artPath(), 'premiumize.png'))

        if control.setting('realdebridusername') != '' and control.setting('realdebridexpirynotice') == 'true':
            account_info = realdebrid.RealDebrid().account_info()
            if account_info:
                FormatDateTime = "%Y-%m-%dT%H:%M:%S.%fZ"
                try: 
                    expires = datetime.strptime(account_info['expiration'], FormatDateTime)
                except:
                    try: 
                        expires = datetime(*(time.strptime(account_info['expiration'], FormatDateTime)[0:6]))
                    except:
                        expires = datetime.today()-timedelta(days=1)
                        control.notification(message='Real-Debrid Account has no expiration. Invalid or free account.', icon=control.joinPath(control.artPath(), 'realdebrid.png'))
                days_remaining = (expires - datetime.today()).days
                if days_remaining >= 0:
                    if self.withinRangeCheck('realdebrid', days_remaining):
                        control.notification(message='Real-Debrid Account expires in %s days' % days_remaining, icon=control.joinPath(control.artPath(), 'realdebrid.png'))

    def withinRangeCheck(self, debrid_provider, days_remaining):
        if days_remaining < 15:
            try: 
                current_notification_range = int(control.setting('%s.notification.range' % debrid_provider))
            except: 
                current_notification_range = 5
            for index, day_range in enumerate(self.duration):
                if day_range[0] > days_remaining > day_range[1] and current_notification_range != index:
                    control.setSetting('%s.notification.range' % debrid_provider, str(index))
                    return True
            return False
        else:
            control.setSetting('%s.notification.range' % debrid_provider, '')
            return False

def main():
    while not control.monitor.abortRequested():
        control.log('[ plugin.video.dynasty ]  Service Started', LOGINFO)
        schedTrakt = None
        libraryService = None
        favouritesMonitor = None
        CheckSettingsFile().run()
        SyncMyAccounts().run()
        PremAccntNotification().run()
        ReuseLanguageInvokerCheck().run()
        SyncMovieLibrary().run()
        # Only run initial favourites sync if backup is enabled
        if control.setting('favourites.backup.enable') == 'true':
            FavouritesSync().sync_favourites()
        if control.setting('library.service.update') == 'true':
            libraryService = Thread(target=LibraryService().run)
            libraryService.start()
        if control.setting('general.checkAddonUpdates') == 'true':
            AddonCheckUpdate().run()
        VersionIsUpdateCheck().run()
        checkAutoStart().run()

        syncTraktService = Thread(target=SyncTraktService().run)
        syncTraktService.start()
        
        # Only start favourites monitor if backup is enabled
        if control.setting('favourites.backup.enable') == 'true':
            favouritesMonitor = Thread(target=FavouritesMonitor().run)
            favouritesMonitor.start()

        break
    SettingsMonitor().waitForAbort()
    control.log('[ plugin.video.dynasty ]  Settings Monitor Service Stopping...', LOGINFO)
    del syncTraktService
    control.log('[ plugin.video.dynasty ]  Trakt Sync Service Stopping...', LOGINFO)
    if libraryService:
        del libraryService
        control.log('[ plugin.video.dynasty ]  Library Update Service Stopping...', LOGINFO)
    if favouritesMonitor:
        del favouritesMonitor
        control.log('[ plugin.video.dynasty ]  Favourites Monitor Service Stopping...', LOGINFO)
    if schedTrakt:
        del schedTrakt
    control.log('[ plugin.video.dynasty ]  Service Stopped', LOGINFO)

main()